/**
 * noteクリエイターダッシュボード - フロントエンドスクリプト
 */

document.addEventListener('DOMContentLoaded', function() {
    const searchForm = document.getElementById('creator-search-form');
    const resultsSection = document.getElementById('results-section');
    const loadingElement = document.getElementById('loading');
    const errorMessage = document.getElementById('error-message');
    const creatorInfo = document.getElementById('creator-info');
    const statsSection = document.getElementById('stats-section');
    const articlesSection = document.getElementById('articles-section');
    
    if (searchForm) {
        searchForm.addEventListener('submit', handleSearch);
    }
    
    /**
     * 検索フォームの送信処理
     * @param {Event} event 
     */
    async function handleSearch(event) {
        event.preventDefault();
        
        const creatorId = document.getElementById('creator-id').value.trim();
        const csrfToken = document.querySelector('input[name="csrf_token"]').value;
        
        if (!creatorId) {
            showError('クリエイターIDを入力してください。');
            return;
        }
        
        // UIをリセット
        hideAllResults();
        showLoading(true);
        resultsSection.style.display = 'block';
        
        try {
            const response = await fetch('api/fetch_creator_data.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    creator_id: creatorId,
                    csrf_token: csrfToken
                })
            });
            
            const data = await response.json();
            
            if (!response.ok) {
                throw new Error(data.error || 'データの取得に失敗しました。');
            }
            
            if (data.success) {
                displayResults(data);
            } else {
                throw new Error(data.error || 'データの取得に失敗しました。');
            }
        } catch (error) {
            showError(error.message);
        } finally {
            showLoading(false);
        }
    }
    
    /**
     * 結果を表示
     * @param {Object} data APIレスポンスデータ
     */
    function displayResults(data) {
        // クリエイター情報を表示
        if (data.creator) {
            displayCreatorInfo(data.creator);
        }
        
        // 統計情報を表示
        if (data.stats) {
            displayStats(data.stats);
        }
        
        // 記事一覧を表示
        if (data.articles && data.articles.length > 0) {
            displayArticles(data.articles);
        }
    }
    
    /**
     * クリエイター情報を表示
     * @param {Object} creator クリエイター情報
     */
    function displayCreatorInfo(creator) {
        const avatarElement = document.getElementById('creator-avatar');
        const nameElement = document.getElementById('creator-name');
        const descriptionElement = document.getElementById('creator-description');
        
        if (creator.avatar_url) {
            avatarElement.src = creator.avatar_url;
            avatarElement.alt = creator.name + 'のアバター';
        } else {
            avatarElement.src = 'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"%3E%3Ccircle cx="50" cy="50" r="50" fill="%23ddd"/%3E%3Ctext x="50" y="55" text-anchor="middle" fill="%23999" font-size="40"%3E?%3C/text%3E%3C/svg%3E';
            avatarElement.alt = 'デフォルトアバター';
        }
        
        nameElement.textContent = creator.name || creator.urlname;
        descriptionElement.textContent = creator.description || '';
        
        creatorInfo.style.display = 'block';
    }
    
    /**
     * 統計情報を表示
     * @param {Object} stats 統計情報
     */
    function displayStats(stats) {
        document.getElementById('total-articles').textContent = formatNumber(stats.total_articles);
        document.getElementById('total-likes').textContent = formatNumber(stats.total_likes);
        document.getElementById('avg-likes').textContent = formatNumber(stats.avg_likes);
        
        statsSection.style.display = 'block';
    }
    
    /**
     * 記事一覧を表示
     * @param {Array} articles 記事一覧
     */
    function displayArticles(articles) {
        const tbody = document.getElementById('articles-tbody');
        tbody.innerHTML = '';
        
        articles.forEach(function(article) {
            const row = document.createElement('tr');
            
            // タイトル
            const titleCell = document.createElement('td');
            const titleLink = document.createElement('a');
            titleLink.href = article.url;
            titleLink.target = '_blank';
            titleLink.rel = 'noopener noreferrer';
            titleLink.textContent = article.title || '(タイトルなし)';
            titleCell.appendChild(titleLink);
            row.appendChild(titleCell);
            
            // スキ数
            const likesCell = document.createElement('td');
            likesCell.textContent = formatNumber(article.like_count);
            row.appendChild(likesCell);
            
            // 公開日
            const dateCell = document.createElement('td');
            dateCell.textContent = formatDate(article.published_at);
            row.appendChild(dateCell);
            
            tbody.appendChild(row);
        });
        
        articlesSection.style.display = 'block';
    }
    
    /**
     * エラーメッセージを表示
     * @param {string} message エラーメッセージ
     */
    function showError(message) {
        errorMessage.textContent = message;
        errorMessage.style.display = 'block';
    }
    
    /**
     * ローディング表示を切り替え
     * @param {boolean} show 表示するかどうか
     */
    function showLoading(show) {
        loadingElement.style.display = show ? 'block' : 'none';
    }
    
    /**
     * すべての結果表示を非表示にする
     */
    function hideAllResults() {
        errorMessage.style.display = 'none';
        creatorInfo.style.display = 'none';
        statsSection.style.display = 'none';
        articlesSection.style.display = 'none';
    }
    
    /**
     * 数値をフォーマット
     * @param {number} num 数値
     * @returns {string} フォーマットされた文字列
     */
    function formatNumber(num) {
        if (num === null || num === undefined) {
            return '-';
        }
        return num.toLocaleString('ja-JP');
    }
    
    /**
     * 日付をフォーマット
     * @param {string} dateString 日付文字列
     * @returns {string} フォーマットされた日付
     */
    function formatDate(dateString) {
        if (!dateString) {
            return '-';
        }
        
        try {
            const date = new Date(dateString);
            return date.toLocaleDateString('ja-JP', {
                year: 'numeric',
                month: '2-digit',
                day: '2-digit'
            });
        } catch (e) {
            return dateString;
        }
    }
});
